UNIT GLTree;

{ TSceneTree  - A scene editor for the GLScene component. It works at design and
                at run time.
  Version     - 0.2.25
  Last Change - 26 June 1997
  for more information see help file
}

INTERFACE

USES Classes, ComCtrls, CommCtrl, Controls, DsgnIntf, GLScene, Menus, Messages;

CONST // Resource ID's:
      //--- names ---
      StrLightSource    = 2;
      StrSceneRoot      = 3;
      StrLightRoot      = 4;
      StrObjectRoot     = 5;
      StrCameraRoot     = 6;
      StrStockRoot      = 7;
      StrCamera         = 8;

TYPE TSceneTree = CLASS(TAbstractSceneTree)
     PRIVATE
       FGLScene                : TGLScene;
       ObjectNode,
       CameraNode,
       LightsourceNode,
       StockObjectNode         : TTreeNode;
       AddCameraMenuItem,
       AddLightsourceMenuItem,
       AddObjectMenuItem,
       DelObjectMenuItem       : TMenuItem;
       FRMouseDown             : Boolean;
       FDesigner               : TFormDesigner;
       PROCEDURE AddCameraClick(Sender: TObject);
       PROCEDURE AddLightsourceClick(Sender: TObject);
       PROCEDURE AddObjectClick(Sender: TObject);
       PROCEDURE CMDesignHitTest(VAR Message: TCMDesignHitTest); MESSAGE CM_DESIGNHITTEST;
       PROCEDURE CreateMenu;
       PROCEDURE DeleteObjectClick(Sender: TObject);
       FUNCTION  GetNodeFromItem(CONST Item: TTVItem): TTreeNode;
       PROCEDURE SetScene(AScene: TGLScene);
       PROCEDURE WMLButtonDown(VAR Message: TWMLButtonDown); MESSAGE WM_LBUTTONDOWN;
       PROCEDURE WMRButtonDown(VAR Message: TWMRButtonDown); MESSAGE WM_RBUTTONDOWN;
     PROTECTED
       PROCEDURE AddNewNode(AParent: TComposite; AChild: TSceneObject); VIRTUAL;
       PROCEDURE AddNodes(ANode: TTreeNode; AObject: TSceneObject); VIRTUAL;
       FUNCTION  CanEdit(Node: TTreeNode): Boolean; OVERRIDE;
       PROCEDURE CreateWnd; OVERRIDE;
       PROCEDURE ContextMenuPopup(Sender: TObject);
       PROCEDURE DragOver(Source: TObject; X, Y: Integer; State: TDragState; VAR Accept: Boolean); OVERRIDE;
       PROCEDURE DragDrop(Source: TObject; X, Y: Integer); OVERRIDE;
       PROCEDURE Edit(CONST Item: TTVItem); OVERRIDE;
       FUNCTION  GetNodeFromObject(AObject: TSceneObject): TTreeNode; VIRTUAL;
       PROCEDURE Notification(AComponent: TComponent; Operation: TOperation); OVERRIDE;
       PROCEDURE Notify(AObject: TSceneObject; Operation: TSceneOperation); OVERRIDE;
       PROCEDURE ReadScene; VIRTUAL;
       PROCEDURE ResetTree; VIRTUAL;
       PROCEDURE WndProc(VAR Message: TMessage); OVERRIDE;
     PUBLIC
       CONSTRUCTOR Create(AOwner: TComponent); OVERRIDE;
       DESTRUCTOR  Destroy; OVERRIDE;
       PROCEDURE Loaded; OVERRIDE;

       PROPERTY Designer : TFormDesigner READ FDesigner;
       PROPERTY Items;
     PUBLISHED
       PROPERTY Scene: TGLScene READ FGLScene WRITE SetScene;
       PROPERTY Visible;
     END;

//------------------------------------------------------------------------------

IMPLEMENTATION

USES Dialogs, Forms, GLObjects, Graphics, SysUtils, Windows;

{$R GLTree.RES}

VAR LangOffset : Word = 0;

//------------------------------------------------------------------------------

CONSTRUCTOR TSceneTree.Create(AOwner: TComponent);

BEGIN
  INHERITED Create(Aowner);
  FGLScene:=NIL;
  Width:=300;
  Height:=300;
  ShowButtons:=False;
  ShowRoot:=False;
  DragMode:=dmManual;
  HideSelection:=True;
  Images:=ObjectIcons;
END;

//------------------------------------------------------------------------------

DESTRUCTOR TSceneTree.Destroy;

BEGIN
  IF assigned(FGLScene) THEN FGLScene.RemoveNotifier(Self);
  INHERITED Destroy;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.AddCameraClick(Sender: TObject);

BEGIN
  FGLScene.Cameras.AddNewChild(TCamera);
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.AddLightsourceClick(Sender: TObject);

VAR LS : TLightSource;

BEGIN
  LS:=FGLScene.Lightsources.AddNewChild(TLightsource) AS TLightSource;
  LS.Shining:=True;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.AddObjectClick(Sender: TObject);

VAR AParent   : TComposite;
    NewObject : TSceneObject;

BEGIN
  IF assigned(Selected) AND (Selected.Level > 0) THEN
  BEGIN
    // where has the new object to be added
    AParent:=TSceneObject(Selected.Data) AS TComposite;
    NewObject:=AParent.AddNewChild(GetClassFromIndex(TMenuItem(Sender).MenuIndex));
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.CMDesignHitTest(VAR Message: TCMDesignHitTest);

BEGIN
  IF assigned(GetNodeAt(Message.XPos,Message.YPos)) THEN Message.Result:=1
                                                    ELSE Message.Result:=0;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.CreateMenu;

// creates a popup menu for adding and deleting scene objects

BEGIN
  // basic items
  AddCameraMenuItem:=NewItem('Add Camera',0,False,True,AddCameraClick,0,'AddCameraItem');
  AddLightsourceMenuItem:=NewItem('Add Lightsource',0,False,True,AddLightsourceClick,0,'AddLightsourceItem');
  AddObjectMenuItem:=NewItem('Add Object',0,False,True,NIL,0,'AddObjectItem');
  DelObjectMenuItem:=NewItem('Delete Object',0,False,True,DeleteObjectClick,0,'DelObjectItem');
  // merge them to a menu
  PopupMenu:=NewPopupMenu(Self,'ObjectMenu',paRight,False,
                          [AddCameraMenuItem,
                           AddLightsourceMenuItem,
                           NewLine,
                           AddObjectMenuItem,
                           NewLine,
                           DelObjectMenuItem]);
  PopupMenu.Alignment:=paLeft;
  PopupMenu.OnPopup:=ContextMenuPopup;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.DeleteObjectClick(Sender: TObject);

VAR AObject      : TSceneObject;
    Allowed,
    KeepChildren : Boolean;
    ConfirmMsg   : STRING;
    Buttons      : TMsgDlgButtons;

BEGIN
  IF assigned(Selected) AND (Selected.Level > 1) THEN
  BEGIN
    AObject:=TSceneObject(Selected.Data);
    // ask for confirming
    ConfirmMsg:='Ok to delete the marked object';
    Buttons:=[mbOK,mbCancel];
    // are there children to care for?
    IF (AObject IS TComposite) AND (TComposite(AObject).Count > 0) THEN
    BEGIN
      ConfirmMsg:=ConfirmMsg+' and ALL its children?';
      Buttons:=[mbAll]+Buttons;
    END
    ELSE ConfirmMsg:=ConfirmMsg+'?';
    CASE MessageDlg(ConfirmMsg,mtConfirmation,Buttons,0) OF
      mrAll    : BEGIN
                   KeepChildren:=False;
                   Allowed:=True;
                 END;
      mrOK     : BEGIN
                   KeepChildren:=True;
                   Allowed:=True;
                 END;
      mrCancel : Allowed:=False;
    END;
    // deletion allowed?
    IF allowed THEN
    BEGIN
      IF AObject IS TComposite THEN AObject.Parent.Remove(AObject,KeepChildren);
      AObject.Free;
      IF csDesigning IN ComponentState THEN Designer.SelectComponent(Self);
    END
  END;
END;

//------------------------------------------------------------------------------

FUNCTION TSceneTree.GetNodeFromItem(CONST Item: TTVItem): TTreeNode;

BEGIN
  WITH Item DO
    IF (state AND TVIF_PARAM) <> 0 THEN Result := Pointer(lParam)
                                   ELSE Result := Items.GetNode(hItem);
end;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.SetScene(AScene: TGLScene);

BEGIN
  IF AScene <> FGLScene THEN
  BEGIN
    IF assigned(FGLScene) THEN
    BEGIN
      FGLScene.RemoveNotifier(Self);
      ResetTree;
    END;
    FGLScene:=AScene;
    IF assigned(FGLScene) THEN
    BEGIN
      FGLScene.AddNotifier(Self);
      ReadScene;
      Items.GetFirstNode.Text:=LoadStr(StrSceneRoot+LangOffset)+' (editing: '+FGLScene.Name+')'
    END
    ELSE Items.GetFirstNode.Text:=LoadStr(StrSceneRoot+LangOffset)+' (no scene)'
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.WMLButtonDown(VAR Message: TWMLButtonDown);

VAR Node : TTreeNode;

BEGIN
  INHERITED;
  Node:=GetNodeAt(Message.XPos,Message.YPos);
  IF assigned(Node) THEN
  BEGIN
    Node.Selected:=True;
    Node.Focused:=True;
    IF Node.Level >= 2 THEN
    BEGIN
      IF (csDesigning IN ComponentState) THEN Designer.SelectComponent(TSceneObject(Node.Data));
      IF DragMode = dmManual THEN BeginDrag(False);
    END;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.WMRButtonDown(VAR Message: TWMRButtonDown);

VAR Node     : TTreeNode;
    PopupPos : TPoint;


BEGIN
  INHERITED;
  Node := GetNodeAt(Message.XPos, Message.YPos);
  IF assigned(Node) THEN
  BEGIN
    Node.Focused := True;
    Node.Selected := True;
    PopupPos:=ClientToScreen(Point(Message.XPos,Message.YPos));
    PopUpMenu.Popup(PopupPos.X+20,PopupPos.Y);
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.ContextMenuPopup(Sender: TObject);

VAR ObjectList : TStringList;
    I          : Integer;

BEGIN
  ObjectList:=TStringList.Create;
  TRY
    IF TSceneObject(Selected.Data) IS TComposite THEN AddObjectMenuItem.enabled:=True
                                                 ELSE AddObjectMenuItem.enabled:=False;
    IF Selected = LightSourceNode THEN AddLightsourceMenuItem.enabled:=True
                                  ELSE AddLightsourceMenuItem.enabled:=False;
    IF Selected = CameraNode THEN AddCameraMenuItem.enabled:=True
                             ELSE AddCameraMenuItem.enabled:=False;
    IF (TSceneObject(Selected.Data) IS TComposite) AND
       Selected.HasAsParent(ObjectNode)            THEN AddObjectMenuItem.enabled:=True
                                                   ELSE AddObjectMenuItem.enabled:=False;
    IF (Selected.Level >= 2) AND
       NOT Selected.HasAsParent(StockObjectNode) THEN DelObjectMenuItem.enabled:=True
                                                 ELSE DelObjectMenuItem.enabled:=False;
    // before popup create a new list of all registered scene objects
    IF AddObjectMenuItem.enabled THEN
    BEGIN
      GetRegisteredSceneObjects(ObjectList);
      WHILE AddObjectMenuItem.Count > 0 DO AddObjectMenuItem.Delete(0);
      FOR I:=0 TO ObjectList.Count-1 DO
        AddObjectMenuItem.Add(NewItem(ObjectList[I],0,False,True,AddObjectClick,0,''));
    END;
  FINALLY
    ObjectList.Free;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.DragOver(Source: TObject; X, Y: Integer; State: TDragState; VAR Accept: Boolean);

VAR SourceObject    : TSceneObject;
    DestinationNode : TTreeNode;

BEGIN
  INHERITED DragOver(Source,X,Y,State,Accept);
  IF Source IS TSceneTree THEN
  BEGIN
    DestinationNode:=GetNodeAt(X,Y);
    SourceObject:=TSceneObject(TSceneTree(Source).Selected.Data);
    // there're many restrictions for acceptance of a drag'n drop operation
    IF assigned(DestinationNode)      AND
       (DestinationNode.Level >= 1)   {AND
       (DestinationNode <> Selected)  AND
       (TSceneObject(DestinationNode.Data) IS TComposite)} THEN
    BEGIN
      Accept:=True;
      // special cases go here
      IF (SourceObject IS TLightsource) AND NOT (DestinationNode.AbsoluteIndex = 1) THEN Accept:=False;
    END;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.DragDrop(Source: TObject; X, Y: Integer);

VAR SourceNode,
    DestinationNode   : TTreeNode;
    SourceObject,
    DestinationObject : TSceneObject;

BEGIN
  INHERITED DragDrop(Source,X,Y);
  DestinationNode:=DropTarget;
  IF assigned(DestinationNode) AND (Source IS TSceneTree) THEN
  BEGIN
    Items.BeginUpdate;
    SourceNode:=TSceneTree(Source).Selected;
    IF SourceNode = DestinationNode THEN Exit;
    SourceObject:=SourceNode.Data;
    DestinationObject:=DestinationNode.Data;
    IF DestinationObject IS TComposite
      THEN TComposite(DestinationObject).Insert(0,SourceObject)
      ELSE DestinationObject.Parent.Insert(DestinationObject.Index,SourceObject);
    Items.EndUpDate;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.Loaded;

BEGIN
  INHERITED Loaded;
  IF csDesigning IN ComponentState THEN
  BEGIN
    FullCollapse;
    Items.GetFirstNode.Expand(False);
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.ResetTree;

// delete all subtrees (empty tree)

BEGIN
  Items.BeginUpdate;
  LightsourceNode.DeleteChildren; LightsourceNode.Data:=NIL;
  CameraNode.DeleteChildren;      CameraNode.Data:=NIL;
  ObjectNode.DeleteChildren;      ObjectNode.Data:=NIL;
  Items.EndUpdate;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.WndProc(var Message: TMessage);

// this is a very tricky way to get drag messages at design time

BEGIN
  IF Message.Msg = CM_DRAG THEN Dispatch(Message)
                           ELSE INHERITED WndProc(Message);
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.ReadScene;

VAR I : Integer;
  
BEGIN
  Items.BeginUpdate;
  WITH FGLScene DO
  BEGIN
    IF assigned(LightSources) THEN
    BEGIN
      LightSourceNode.Data:=LightSources;
      WITH LightSources DO
        FOR I:=0 TO Count-1 DO AddNodes(LightsourceNode,Children[I]);
    END;
    IF assigned(Cameras) THEN
    BEGIN
      CameraNode.Data:=Cameras;
      WITH Cameras DO
        FOR I:=0 TO Count-1 DO AddNodes(CameraNode,Children[I]);
    END;
    IF assigned(Objects) THEN
    BEGIN
      ObjectNode.Data:=Objects;
      WITH Objects DO
        FOR I:=0 TO Count-1 DO AddNodes(ObjectNode,Children[I]);
    END;
  END;
  Items.EndUpdate;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.Edit(CONST Item: TTVItem);

// used as notification in case the user wants to change a SceneObject's name

VAR Node    : TTreeNode;
    OldName : STRING;

BEGIN
  INHERITED Edit(Item);
  Node:=GetNodeFromItem(Item);
  IF assigned(Node) AND assigned(Node.Data) THEN
  TRY
    OldName:=TSceneObject(Node.Data).Name;
    TSceneObject(Node.Data).Name:=Node.Text;
  EXCEPT
    Node.Text:=OldName;
    RAISE;
  END;
END;

//------------------------------------------------------------------------------

FUNCTION FindObjectNode(ANode: TTreeNode; AObject: TSceneObject) : TTreeNode;

VAR I : Integer;

BEGIN
  Result:=NIL;
  IF assigned(ANode)      AND
     assigned(ANode.Data) AND
     (TSceneObject(ANode.Data) = AObject) THEN Result:=ANode
                                          ELSE
    FOR I:=0 TO ANode.Count-1 DO
    BEGIN
      Result:=FindObjectNode(ANode.Item[I],AObject);
      IF assigned(Result) THEN Break;
    END;
END;

//------------------------------------------------------------------------------

FUNCTION TSceneTree.GetNodeFromObject(AObject: TSceneObject): TTreeNode;

VAR Root : TTreeNode;

BEGIN
  // first try the root nodes
  IF AObject = TSceneObject(LightsourceNode.Data) THEN Result:=LightsourceNode ELSE
    IF AObject = TSceneObject(CameraNode.Data) THEN Result:=CameraNode ELSE
      IF AObject = TSceneObject(ObjectNode.Data) THEN Result:=ObjectNode ELSE
        IF AObject = TSceneObject(StockObjectNode.Data) THEN Result:=StockObjectNode ELSE
  // now search the branches
  IF AObject IS TLightSource THEN Result:=FindObjectNode(LightsourceNode,AObject)
                             ELSE
    IF AObject IS TCamera THEN Result:=FindObjectNode(CameraNode,AObject)
                          ELSE
    BEGIN
      Result:=FindObjectNode(ObjectNode,AObject);
      IF Result = NIL THEN Result:=FindObjectNode(StockObjectNode,AObject);
    END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.Notification(AComponent: TComponent; Operation: TOperation);

BEGIN
  INHERITED Notification(AComponent,Operation);
  // Do we currently edit a scene?
  IF FGLScene = NIL THEN Exit; // no -> get out of here
  // handle deletion
  IF NOT (csDestroying IN ComponentState) AND (Operation = opRemove)THEN
  BEGIN
    IF AComponent = FGLScene  THEN
    BEGIN
      FGLScene:=NIL;
      Items.GetFirstNode.Text:=LoadStr(StrSceneRoot+LangOffset)+' (no scene)';
      ResetTree;
    END;
    IF (AComponent IS TSceneObject) AND (TSceneObject(AComponent).Scene = FGLScene) THEN
    BEGIN
    END;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.Notify(AObject: TSceneObject; Operation: TSceneOperation);

VAR ANode,
    Parentnode : TTreeNode;

BEGIN
  ANode:=GetNodeFromObject(AObject);
  CASE Operation OF
    soRename : IF assigned(ANode) THEN ANode.Text:=AObject.Name;
    soAdd    : AddNewNode(AObject.Parent,AObject);
    soRemove : IF assigned(ANode) THEN ANode.Delete;
  END;
END;

//------------------------------------------------------------------------------

{:Adds or inserts a newly inserted scene object ('AChild')
 into the tree hierarchy below 'AParent'.}

PROCEDURE TSceneTree.AddNewNode(AParent: TComposite; AChild: TSceneObject);

VAR OtherNode,
    CurrentNode : TTreeNode;
    I           : Integer;

BEGIN
  OtherNode:=GetNodeFromObject(AParent);
  IF assigned(OtherNode) THEN
  BEGIN
    IF AParent.Count-1 <= AChild.Index
      THEN CurrentNode:=Items.AddChildObject(OtherNode,AChild.Name,AChild)
      ELSE
      BEGIN
        // special handling for insert needed
        OtherNode:=GetNodeFromObject(AParent[AChild.Index+1]);
        CurrentNode:=Items.InsertObject(OtherNode,AChild.Name,AChild);
      END;
      IF AChild IS TComposite THEN
      WITH AChild AS TComposite DO
        FOR I:=0 TO Count-1 DO AddNodes(CurrentNode,Children[I]);
    WITH CurrentNode DO
    BEGIN
      MakeVisible;
      ImageIndex:=GetImageIndex(TSceneObjectClass(AChild.ClassType));
      SelectedIndex:=ImageIndex;
    END;
  END;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.AddNodes(ANode: TTreeNode; AObject: TSceneObject);

VAR CurrentNode : TTreeNode;
    I           : Integer;

BEGIN
  CurrentNode:=Items.AddChildObject(ANode,AObject.Name,AObject);
  CurrentNode.ImageIndex:=GetImageIndex(TSceneObjectClass(AObject.ClassType));
  CurrentNode.SelectedIndex:=CurrentNode.ImageIndex;
  IF AObject IS TComposite THEN
  WITH AObject AS TComposite DO
    FOR I:=0 TO Count-1 DO AddNodes(CurrentNode,Children[I]);
END;

//------------------------------------------------------------------------------

FUNCTION TSceneTree.CanEdit(Node: TTreeNode): Boolean;

BEGIN
  INHERITED CanEdit(Node);
  IF Node.Level > 1 THEN Result:=True
                    ELSE Result:=False;
END;

//------------------------------------------------------------------------------

PROCEDURE TSceneTree.CreateWnd;

VAR CurrentNode : TTreeNode;
    I           : Integer;

BEGIN
  INHERITED CreateWnd;
  WITH Items DO
  BEGIN
    BeginUpdate;
    // first add the scene root
    CurrentNode:=Add(NIL,LoadStr(StrSceneRoot+LangOffset)+' (no scene)');
    CurrentNode.ImageIndex:=SceneRootIndex;
    CurrentNode.SelectedIndex:=SceneRootIndex;
    // then the root for all lightsources
    LightsourceNode:=AddChild(GetFirstNode,LoadStr(StrLightRoot+LangOffset));
    LightsourceNode.ImageIndex:=LightsourceRootIndex;
    LightsourceNode.SelectedIndex:=LightsourceRootIndex;
    // next the root for all cameras
    CameraNode:=AddChild(GetFirstNode,LoadStr(StrCameraRoot+LangOffset));
    CameraNode.ImageIndex:=CameraRootIndex;
    CameraNode.SelectedIndex:=CameraRootIndex;
    // and the root for all objects
    ObjectNode:=AddChild(GetFirstNode,LoadStr(StrObjectRoot+LangOffset));
    ObjectNode.ImageIndex:=ObjectRootIndex;
    ObjectNode.SelectedIndex:=ObjectRootIndex;
    // finally the root for all stock objects
    StockObjectNode:=AddChild(GetFirstNode,LoadStr(StrStockRoot+LangOffset));
    StockObjectNode.ImageIndex:=StockObjectRootIndex;
    StockObjectNode.SelectedIndex:=StockObjectRootIndex;
    IF assigned(ObjectStock) THEN
    BEGIN
      StockObjectNode.Data:=ObjectStock;
      WITH ObjectStock DO
        FOR I:=0 TO Count-1 DO AddNodes(StockObjectNode,Children[I]);
    END;
    EndUpdate;
  END;

  // create a popup menu
  CreateMenu;
  // at design time we use the owners designer interface
  IF csDesigning IN ComponentState THEN FDesigner:=TForm(Owner).Designer AS TFormDesigner;
END;

//------------------------------------------------------------------------------

BEGIN
  // determine user default language for localized messages
  CASE GetUserDefaultLangID AND $3FF OF
    LANG_GERMAN  : LangOffset:=1000;
    LANG_ITALIAN : LangOffset:=2000
  ELSE LangOffset:=0;
  END;
  LangOffset:=0;
END.
